/*=================================================================
 *
 * mgf_intAJD2_c.c ...  For fixed set of parameters, calculate moment-generating function for multiple horizons simultaneously.
 *                      This function is vectorized.
 *
 * The calling syntax is:
 *
 *		p = mgf_intAJD2_c(x0, k, theta, sigma, L, mu, q, T)
 *
 * x0       ... initial value of AJD
 * k        ... mean reversion speed
 * theta    ... mean reversion level
 * L        ... jump intensity
 * mu       ... expected jump size of exponentially distributed jumps
 * T        ... time horizons (vector)
 * q        ... value at which to evaluate moment generating function (-1 for survuval probabilities)
 *
 *		p = mgf_intAJD2_c(x0, k, theta, sigma, L, mu, q, horizons)
 *
 * sample: mgf_intAJD2_c([0.01; 0.01], [0.1 0.25], [0.02; 0.02], [0.05; 0.05], [0.02; 0.02], [0.03; 0.03], [-1; -1], [1 5; 2 5])
 *
 *=================================================================*/

#include <math.h>
#include "mex.h"

#ifndef max
#define max(a,b) (((a) > (b)) ? (a) : (b))
#endif

/* Recursive computation of loss distribution */
static void mgf_intAJD2_c(double x0[], double horizons[], double k[], double theta[], double sigma[], double L[], double mu[], double p[], double q[], unsigned int num_dates, unsigned int num_horizons)
{
    int i, j, index;
    double gamma, c1, c2, d1, d2, A, B, T;
    double tmp, tmp2, tmp3, tmp4, tmp5, tmp6, tmp8;

    /* For each horizon and each data, calculate survival probability */
    for (i=0; i<num_dates; i++)
    {
        /* For numerical stability, require sigma to be larger than 1e-5 */
        sigma[i] = max(sigma[i], 1e-5);
        
        /* Calculate a couple of auxiliary variables */
        gamma = sqrt(k[i]*k[i] - 2*sigma[i]*sigma[i]*q[i]);
        c1 = (gamma + k[i]) / (2*q[i]);
        c2 = 1 - mu[i]/c1;
        d1 = (-k[i] + gamma) / (2*q[i]); 
        d2 = (d1+mu[i])/c1;
        tmp = (-2*k[i]*theta[i]) / (sigma[i]*sigma[i]);
        tmp2 = L[i]*(c2*d1/c1-d2) / (-gamma*c2*d2);
        tmp3 = L[i]/c2-L[i];
        tmp4 = k[i]*theta[i]/c1;
        tmp5 = tmp3+tmp4;

        for (j=0; j<num_horizons; j++) 
        {
            index = i+j*num_dates;
            T = horizons[index];
            tmp6 = exp(-gamma*T);
            tmp8 = c1+d1*tmp6;
            A = tmp * log( tmp8/(c1 + d1) ) + tmp2 * log( (c2+d2*tmp6)/(c2+d2) ) + tmp5*T ;
            B = (1-tmp6) / tmp8;
            p[index] = exp( A + x0[i] * B);
        }
    }
    return;
}


/* Gateway routine (to Matlab) */
void mexFunction( int nlhs, mxArray *plhs[], 
		  int nrhs, const mxArray*prhs[] )
     
{ 
    double *x0, *p, *horizons; 
    double *k, *theta, *sigma, *L, *mu, *q;
    unsigned int num_dates, num_horizons; 

    /* Get number of companies and number of points in time */
    num_dates = mxGetM(prhs[0]);
    num_horizons = mxGetN(prhs[7]);
    
    /* Assign pointers to the input variables parameters */ 
    x0 = mxGetPr(prhs[0]);
    k = mxGetPr(prhs[1]);
    theta = mxGetPr(prhs[2]);
    sigma =mxGetPr(prhs[3]);
    L = mxGetPr(prhs[4]);
    mu = mxGetPr(prhs[5]);
    q = mxGetPr(prhs[6]);
    horizons = mxGetPr(prhs[7]);
    
    /* Create a mtrix for the return argument */ 
    plhs[0] = mxCreateDoubleMatrix(num_dates, num_horizons, mxREAL);
    p = mxGetPr(plhs[0]);
        
    /* Do the actual computations in a subroutine */
    mgf_intAJD2_c(x0, horizons, k, theta, sigma, L, mu, p, q, num_dates, num_horizons);
    /* p[0] = start[0]; */
    /* p[1] = start[1]; */
    return;
}
